<?php

namespace App\Support;

use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;

class Api
{
    /** Get CRM connection defined in .env / config/database.php */
    public static function crm()
    {
        $connectionName = config('database.crm_connection', env('CRM_CONNECTION', 'crm'));
        return DB::connection($connectionName);
    }

    /** Measure execution time (returns [result, duration_ms]) */
    public static function timed(callable $fn): array
    {
        $t0 = microtime(true);
        $result = $fn();
        $t1 = microtime(true);
        return [$result, (int) round(($t1 - $t0) * 1000)];
    }

    /** Build starting audit payload */
    public static function auditStart(Request $request): array
    {
        return [
            'event'        => 'delivery.update', // you can set this per controller if you want
            'endpoint'     => $request->path(),
            'http_method'  => $request->method(),
            'started_at'   => now('Asia/Dubai')->format('Y-m-d H:i:s'),
            'client_ip'    => $request->ip(),
            'user_agent'   => $request->userAgent(),
            'headers'      => $request->headers->all(),
            'query'        => $request->query(),
            'request_body' => $request->all(),
            'country_code' => $request->query('country_code'),
        ];
    }

    /** Finish audit payload with result+duration */
    public static function auditFinish(array &$audit, float $startedAt, string $result): void
    {
        $endedAt = microtime(true);
        $audit['result']      = $result;
        $audit['ended_at']    = now('Asia/Dubai')->format('Y-m-d H:i:s');
        $audit['duration_ms'] = (int) round(($endedAt - $startedAt) * 1000);
    }

    /** Write audit to logs DB (env-driven named connection) */
    public static function auditWrite(array $payload): void
    {
        try {
            $conn = config('database.logs_connection', env('LOGS_CONNECTION', 'logs_mssql'));
            $db = DB::connection($conn);

            $json = json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

            $db->table('ApiAuditLogs')->insert([
                'Endpoint'     => substr((string) Arr::get($payload, 'endpoint', ''), 0, 255),
                'ClientIP'     => substr((string) Arr::get($payload, 'client_ip', ''), 0, 64),
                'UserAgent'    => substr((string) Arr::get($payload, 'user_agent', ''), 0, 512),
                'Result'       => substr((string) Arr::get($payload, 'result', ''), 0, 64),
                'StartedAt'    => Arr::get($payload, 'started_at'),
                'EndedAt'      => Arr::get($payload, 'ended_at'),
                'DurationMs'   => (int) Arr::get($payload, 'duration_ms', 0),
                'DbConnectMs'  => (int) Arr::get($payload, 'db_connect_ms', 0),
                'ExistCheckMs' => (int) Arr::get($payload, 'exist_check_ms', 0),
                'UpdateMs'     => (int) Arr::get($payload, 'update_ms', 0),
                'RowsAffected' => (int) Arr::get($payload, 'rows_affected', 0),
                'PayloadJson'  => $json,
            ]);
        } catch (\Throwable $e) {
            \Log::error('ApiAuditLogs write failed', ['err' => $e->getMessage()]);
        }
    }
}
