<?php

namespace App\Services;

use Firebase\JWT\JWT;
use Firebase\JWT\Key;

class JwtService
{
    /**
     * Returns decoded payload array on success; throws \RuntimeException on failure.
     */
    public function verifyBearerToken(string $authorizationHeader, ?string $secret = null): array
    {
        if (!preg_match('/^Bearer\s+(.+)$/i', $authorizationHeader ?? '', $m)) {
            throw new \RuntimeException('wrong credential');
        }

        $token  = $m[1];
        $secret = $secret ?: env('JWT_SECRET', config('app.key'));

        try {
            if (class_exists(\Firebase\JWT\JWT::class) && class_exists(\Firebase\JWT\Key::class)) {
                $decoded = JWT::decode($token, new Key($secret, 'HS256'));
                return (array) $decoded;
            }

            $payload = $this->jwtDecodeHs256($token, $secret);
            if ($payload === false) {
                throw new \RuntimeException('wrong credential');
            }
            return $payload;
        } catch (\Throwable $e) {
            throw new \RuntimeException('wrong credential');
        }
    }

    /**
     * Minimal manual HS256 decode/verify (fallback).
     */
    private function jwtDecodeHs256(string $jwt, string $secret)
    {
        $parts = explode('.', $jwt);
        if (count($parts) !== 3) return false;

        [$h, $p, $s] = $parts;

        $sig = hash_hmac('sha256', $h . '.' . $p, $secret, true);
        $sigB64 = rtrim(strtr(base64_encode($sig), '+/', '-_'), '=');
        if (!hash_equals($sigB64, $s)) return false;

        $payloadJson = base64_decode(strtr($p, '-_', '+/'));
        $payload = json_decode($payloadJson, true);

        $now = time();
        if (isset($payload['nbf']) && $payload['nbf'] > $now) return false;
        if (isset($payload['exp']) && $payload['exp'] < $now) return false;

        return $payload ?? false;
    }
}
