<?php

namespace App\Services;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Arr;

class AuditLogger
{
    private array $payload = [];
    private string $connection;

    public function __construct(?string $connection = null)
    {
        $this->connection = $connection ?? config('database.logs_connection', 'logs_mssql');
    }

    public function start(Request $request): void
    {
        $this->payload = [
            'endpoint'     => $request->path(),
            'http_method'  => $request->method(),
            'started_at'   => now('Asia/Dubai')->format('Y-m-d H:i:s'),
            'client_ip'    => $request->ip(),
            'user_agent'   => $request->userAgent(),
            'headers'      => $request->headers->all(),
            'query'        => $request->query(),
            'request_body' => $request->all(),
            'country_code' => $request->query('country_code'),
        ];
    }

    public function success(float $startedAt, array $metrics = []): void
    {
        $this->finishPayload('success', $startedAt, $metrics);
        $this->write();
    }

    public function notFound(float $startedAt, array $metrics = []): void
    {
        $this->finishPayload('not_found', $startedAt, $metrics);
        $this->write();
    }

    public function error(float $startedAt, \Throwable $e): void
    {
        $this->finishPayload('exception', $startedAt, [
            'exception_class'   => get_class($e),
            'exception_message' => $e->getMessage(),
        ]);
        $this->write();
    }

    private function finishPayload(string $result, float $startedAt, array $extra = []): void
    {
        $endedAt = microtime(true);
        $this->payload = array_merge($this->payload, [
            'result'      => $result,
            'ended_at'    => now('Asia/Dubai')->format('Y-m-d H:i:s'),
            'duration_ms' => (int) round(($endedAt - $startedAt) * 1000),
        ], $extra);
    }

    private function write(): void
    {
        try {
            $db = DB::connection($this->connection);

            $json = json_encode($this->payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

            $db->table('ApiAuditLogs')->insert([
                'Endpoint'     => substr((string)Arr::get($this->payload, 'endpoint', ''), 0, 255),
                'ClientIP'     => substr((string)Arr::get($this->payload, 'client_ip', ''), 0, 64),
                'UserAgent'    => substr((string)Arr::get($this->payload, 'user_agent', ''), 0, 512),
                'Result'       => substr((string)Arr::get($this->payload, 'result', ''), 0, 64),
                'StartedAt'    => Arr::get($this->payload, 'started_at'),
                'EndedAt'      => Arr::get($this->payload, 'ended_at'),
                'DurationMs'   => (int) Arr::get($this->payload, 'duration_ms', 0),
                'DbConnectMs'  => (int) Arr::get($this->payload, 'db_connect_ms', 0),
                'ExistCheckMs' => (int) Arr::get($this->payload, 'exist_check_ms', 0),
                'UpdateMs'     => (int) Arr::get($this->payload, 'update_ms', 0),
                'RowsAffected' => (int) Arr::get($this->payload, 'rows_affected', 0),
                'PayloadJson'  => $json,
            ]);
        } catch (\Throwable $e) {
            \Log::error('ApiAuditLogs write failed', ['err' => $e->getMessage()]);
        }
    }
}
