<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use App\Support\Api; // <--  helper
use Exception;

class DeliveryController extends Controller
{
    public function updateDeliveryConfirmations(Request $request): JsonResponse
    {
        try
        {
            // Sanatize the Inputs  and validation + normalization
            $data = $request->validate([
                'order_id'      => 'required|string',
                'response'      => 'required|string|in:Yes,No,yes,no,YES,NO',
                'delivery_date' => 'required|date_format:Y-m-d',
            ]);
            $orderNumber   = trim($data['order_id']);
            $responseInput = ucfirst(strtolower(trim($data['response']))); // Yes/No
            $deliveryDate  = trim($data['delivery_date']);
            $countryCode   = $request->query('country_code');

            // Connect to CRM with timing and delay status
            [$db, $connMs] = Api::timed(fn () => Api::crm());
            $audit['db_connect_ok'] = true;
            $audit['db_connect_ms'] = $connMs;

            // ----Function to fetch CRM DB Query  confirm_delivery
            $base = fn() => $this->confirmDeliveryQuery($db, $orderNumber, $deliveryDate, $countryCode);

            // Existence check (timed)
            [$exists, $existMs] = Api::timed(fn () => $base()->exists());
            $audit['exist_check_ms'] = $existMs;

            if (!$exists)
            {
                return response()->json(
                    [
                        'status'  => false,
                        'message' => 'Sorry, could not find any records for this order in Delivery confirmation Table',
                    ], 404);
            }

            // Update (timed)
            $updateData = [
                'response'    => $responseInput,
                'response_on' => now('Asia/Dubai')->format('Y-m-d H:i:s'),
            ];
            if ($responseInput === 'No')
            {
                $updateData['notes'] = 'No';
            }

            [$affected, $updateMs] = Api::timed(fn () => $base()->update($updateData));
            $audit['update_ms']     = $updateMs;
            $audit['rows_affected'] = (int) $affected;

            return response()->json(
                [
                    'status'  => true,
                    'message' => 'Delivery Confirmation Updated successfully',
                ], 200);

        }
        catch (Exception $e)
        {
            // Preserve your current semantics: return e->getMessage() with 200
            if (!isset($audit['db_connect_ok']))
            {
                $audit['db_connect_ok']    = false;
                $audit['db_connect_error'] = $e->getMessage();
            }
            $audit['exception_class']   = get_class($e);
            $audit['exception_message'] = $e->getMessage();
            return response()->json($e->getMessage(), 200);
        }
    }

    /**
     * CRM DB Query.
     */
    private function confirmDeliveryQuery($db, string $orderNumber, string $deliveryDate, ?string $countryCode)
    {
        return $db->table('confirm_delivery')
            ->where('orno', $orderNumber)
            ->when($countryCode, fn($q) => $q->where('company', $countryCode))
            ->whereDate('delivery_date', $deliveryDate);
    }
}
