from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy import select
from app.database.connection import SessionLocal
from app.models.web_order import x2_web_cc_orders
from app.schemas.web_order_schema import WebCCOrder
from app.services.token_helper import verify_jwt_token

router = APIRouter()

def map_country_code(code: str) -> str:
    return {
        "777": "UAE",
        "200": "KSA",
        "100": "OMAN",
        "800": "QATAR"
    }.get(code, None)

@router.get("/click-collect-orders", response_model=list[WebCCOrder], tags=["Click & Collect Orders"])
def get_web_orders(current_user: str = Depends(verify_jwt_token)):
    if current_user != "ayaz":
        raise HTTPException(status_code=403, detail="Access denied.")

    db = SessionLocal()
    try:
        stmt = select(x2_web_cc_orders)
        rows = db.execute(stmt).mappings().all()

        response = []
        for row in rows:
            try:
                response.append({
                    "Web_Order": row.get("Web_Order"),
                    "Customer_Phone": row.get("Customer_Phone"),
                    "Order_Date": row.get("Order_Date"),
                    "Order_TotalAmount": float(row.get("Order_TotalAmount") or 0),
                    "Collect_Date": row.get("Collect_Date"),
                    "Store_ID": row.get("Store_ID"),
                    "Country_Code": map_country_code(str(row.get("Country_Code")) if row.get("Country_Code") else None),
          "Order_Status": str(row.get("Order_Status")) if row.get("Order_Status") is not None else None,
                    "actual_collect_date": row.get("actual_collect_date"),
                    "orderReadyDate": row.get("orderReadyDate"),
                })
            except Exception as e:
                print("❌ Failed to parse row:", row)
                print("❗ Error:", e)

        return response
    except Exception as e:
        print("🔥 Fatal error in /click-collect-orders:", e)
        raise HTTPException(status_code=500, detail="Internal Server Error")
    finally:
        db.close()
